import argparse
import getpass
import os
import sys
import xml.etree.ElementTree as ElementTree

from contextlib import contextmanager
from ncclient import manager
from ncclient.transport.errors import AuthenticationError
from ncclient.transport.errors import TransportError
from ncclient.xml_ import to_ele
from tabulate import tabulate
from zeroconf import IPVersion

from network_dev_config.listener import discover_switch_devices
from network_dev_config.version import __version__


def enabledstr(b):
    if b:
        return "enabled"
    else:
        return "disabled"


def get_rstp_setting(manager, source):
    startup_config = manager.get_config(source=source).data_xml

    ns = {
        'oc-stp': 'http://openconfig.net/yang/spanning-tree',
        'oc-stp-types': 'http://openconfig.net/yang/spanning-tree/types',
    }

    startup = ElementTree.fromstring(startup_config)
    # get the stp configuration
    rstp_enabled = False
    stp = startup.find("oc-stp:stp", ns)
    if stp is not None:
        enabled_protocols = stp.find("oc-stp:global/oc-stp:config/oc-stp:enabled-protocol", ns)
        if enabled_protocols is not None and enabled_protocols.text == "oc-stp-types:RSTP":
            rstp_enabled = True

    return rstp_enabled


def edit_rstp_setting(manager, target, new_state):
    if new_state:
        op = "merge"
    else:
        op = "remove"

    # construct the edit-config operation
    modification = """\
<config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
<stp xmlns="http://openconfig.net/yang/spanning-tree"
     xmlns:oc-stp-types="http://openconfig.net/yang/spanning-tree/types">
    <global>
        <config>
            <enabled-protocol nc:operation="{op}">oc-stp-types:RSTP</enabled-protocol>
        </config>
    </global>
</stp>
</config>
    """.format(op=op)

    manager.edit_config(target=target, config=modification)


def edit_pw_setting(manager, target, new_pw):

    # construct the edit-config operation
    modification = """\
<config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0">
   <system xmlns="urn:ietf:params:xml:ns:yang:ietf-system">
      <authentication>
         <user>
            <name>admin</name>
            <password nc:operation="merge">$0${pw}</password>
         </user>
      </authentication>
   </system>
</config>
    """.format(pw=new_pw)

    manager.edit_config(target=target, config=modification)


def system_restart(args):
    # ncclient's interface is ill suited for reboot command
    # in its exit code, it will try to close session gracefully and there is no way to opt out
    # so anticipate the transport error below and in addition verify that the device is
    # actually reset

    m = manager.connect(host=args.hostname,
                        port=830,
                        username=args.username,
                        password=args.password,
                        hostkey_verify=False)
    m.dispatch(to_ele('<sys:system-restart xmlns:sys="urn:ietf:params:xml:ns:yang:ietf-system"/>'))

    try:
        m.close_session()
        raise Exception('Failed to reset {}'.format(args.hostname))
    except TransportError:
        print("{} is being reset".format(args.hostname))


def main(argv=sys.argv[1:]):
    parser = argparse.ArgumentParser(os.path.basename(sys.argv[0]))
    parser.add_argument('hostname', nargs='?', help='hostname of device')
    parser.add_argument('--username', default=None, help='login username')
    parser.add_argument('--password', default=None, help='login password')
    parser.add_argument('--timeout', metavar='SECONDS', default=5, type=int, help='timeout (s) for device discovery')

    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument('--rstp', choices=['query', 'enable', 'disable'],
                       help='get/set Rapid Spanning Tree Protocol state')
    group.add_argument('--discover', action='store_true', help='output hostnames for devices discovered via mDNS')
    group.add_argument('--update_password', metavar='NEW_PASSWORD', help='change password to NEW_PASSWORD')
    group.add_argument('--update-password', help=argparse.SUPPRESS)
    group.add_argument('--query_device_info', action='store_true', help='query device information')
    group.add_argument('--query-device-info', action='store_true', help=argparse.SUPPRESS)
    group.add_argument('--reset', action='store_true', help='reset device')
    group.add_argument('--version', action='version', version=__version__)

    args = parser.parse_args(argv)

    if args.discover:
        devs = discover_switch_devices(args.timeout)
        if not devs:
            print('No devices found')
            return

        print('Found the following devices:')
        tab = [[d.server.replace('.local.', ''),
                ', '.join(d.parsed_addresses(IPVersion.V4Only)),
                ', '.join(d.parsed_addresses(IPVersion.V6Only))]
               for d in devs]
        print(tabulate(tab, headers=['Device', 'IPv4 Addresses', 'IPv6 Addresses']))
        return
    elif '--timeout' in sys.argv:
        parser.error('--timeout can only be used in combination with --discover')

    if not args.hostname:
        parser.print_help()
        sys.exit(1)

    default_credentials = False
    # use admin/blank as default credentials if not username nor password were specified
    if args.username is None and args.password is None:
        args.username = "admin"
        args.password = ""
        default_credentials = True
    # error if no username but we got a password
    if args.username is None and args.password is not None:
        parser.error('the following arguments are required if --password is set: --username')
    # if a username was specified but a password was not, prompt for password
    if args.username is not None and args.password is None:
        args.password = getpass.getpass()

    if args.reset:
        system_restart(args)
        return

    @contextmanager
    def _connect_default():
        try:
            with manager.connect(host=args.hostname, port=830, username=args.username,
                                 password=args.password, hostkey_verify=False) as m:
                yield m
        except AuthenticationError:
            if not default_credentials:
                raise
            args.username = input('Username: ')
            args.password = getpass.getpass()

            with manager.connect(host=args.hostname, port=830, username=args.username,
                                 password=args.password, hostkey_verify=False) as m:
                yield m

    with _connect_default() as m:
        if args.query_device_info:

            ns = {
                'sys': 'urn:ietf:params:xml:ns:yang:ietf-system',
            }
            startup_config = ElementTree.fromstring(m.get().data_xml)

            if startup_config.find("sys:system-state", ns) is None:
                raise Exception("missing sys:system-state in response from device")
            if startup_config.find("sys:system-state/sys:platform", ns) is None:
                raise Exception("missing sys:system-state/sys:platform in response from device")

            machine = startup_config.find("sys:system-state/sys:platform/sys:machine", ns)
            if machine is None:
                raise Exception("missing sys:system-state/sys:platform/sys:machine"
                                " in response from device")
            else:
                print("model name: {}".format(machine.text))

            os_version = startup_config.find("sys:system-state/sys:platform/sys:os-version", ns)
            if os_version is None:
                raise Exception("missing sys:system-state/sys:platform/sys:os-version"
                                " in response from device")
            else:
                print("firmware version: {}".format(os_version.text))

            return

        # modify this configuration
        if args.rstp:

            print("RSTP Configuration for %s" % args.hostname)
            print("-----------------------------------")

            current_setting = get_rstp_setting(m, source='startup')
            print("current startup setting: %s" % (enabledstr(current_setting)))

            if args.rstp == 'query':
                return

            # copy config to candidate
            m.copy_config(source='startup', target='candidate')

            # modify the setting
            edit_rstp_setting(m, target='candidate', new_state=bool(args.rstp == 'enable'))

            # then copy the candidate to startup
            m.copy_config(source='candidate', target='startup')

            new_setting = get_rstp_setting(m, source='startup')
            print("new startup setting:     %s" % (enabledstr(new_setting)))

        if args.update_password:
            length = len(args.update_password)
            if length < 6 or length > 32:
                print("error: new password length must be between 6 and 32")
                return

            edit_pw_setting(m, target='candidate', new_pw=args.update_password)

            m.copy_config(source='candidate', target='running')

            print("password has been updated")


if __name__ == '__main__':
    main()
